set floppy_fstool "fstool"
set floppy_tosvd "tosvd"
# set floppy_fstool [file join $home "fstool"]
# set floppy_tosvd [file join $home "tosvd"]

########################################################################
# fstoolFloppyTypes() -	Returns the list of floppy OS types.
########################################################################
proc fstoolFloppyTypes {} {
    global floppy_fstool

    lappend cmd $floppy_fstool
    lappend cmd "-l"

    return [eval exec $cmd]
}

########################################################################
# fstoolVersion() -	Returns the version report from fstool
########################################################################
proc fstoolVersion {} {
    global floppy_fstool

    lappend cmd $floppy_fstool
    lappend cmd "-V"

    return [eval exec $cmd]
}

########################################################################
# tosvdVersion() -	Returns the version report from tosvd
########################################################################
proc tosvdVersion {} {
    global floppy_tosvd

    lappend cmd $floppy_tosvd
    lappend cmd "-V"

    return [eval exec $cmd]
}

########################################################################
# tosvdFormatsSupported() - Returns the long list of formats supported.
#                 This list has descriptive text along the "key" that
#                 can be used to tell tosvd how to write.
########################################################################
proc tosvdFormatsSupported {} {
    global floppy_tosvd

    lappend cmd $floppy_tosvd
    lappend cmd "-qv"

    set returnlist [list]

    # NOTE - should have a catch here

    set input [open "|$cmd" "r"]

    while { [gets $input line] >=0 } {

	#
	# the lines that come back here are of the form
	#      FORMAT_ID     DESCRIPTION
	#
	regexp -line {^([^ \t]+)[ \t]+(.+)} $line ignoreMatch format description

	set returnlist [lappend returnlist [list $format "$description ($format)"]]
    }

    return $returnlist
}

########################################################################
# floppyReport() -	Reports on the given floppy file.
########################################################################
proc floppyReport {file option} {
    global floppy_tosvd

    lappend cmd $floppy_tosvd
    lappend cmd "-r"

    if { $option == "description" } {
	lappend cmd "-t"
    }
    if { $option == "check" } {
	lappend cmd "-m"
    }

    lappend cmd $file
    return [eval exec $cmd]
    # note that the -keepnewline was not included...so to punt the newline
    # NEED A CATCH HERE
}

########################################################################
# floppyListOS() -	List all of the files on the given image with
#                       the given filesystem.
########################################################################
proc floppyListOS {file fstype} {
    global floppy_fstool
    global FloppyType

    lappend cmd $floppy_fstool
    lappend cmd "-r"
    #    lappend cmd "-v"        # don't really need a verbose listing
    lappend cmd "-f"
    lappend cmd $file
    lappend cmd "-F"
    lappend cmd $fstype

    catch {eval exec -keepnewline $cmd} output

    return $output
}

########################################################################
# floppyCheckOSAdd() -	Checks that the given files can be added to
#                       the given OS file.  This routine simply checks
#                       for a good return value.  Upon error, the message
#                       from the command is displayed, and this routine
#                       returns an error.
########################################################################
proc floppyCheckOSAdd {osfile ostype filelist} {
    global floppy_fstool
    global FloppyType

    lappend cmd $floppy_fstool
    lappend cmd "-A"
    lappend cmd "-n"
    lappend cmd "-f"
    lappend cmd $osfile
    lappend cmd "-F"
    lappend cmd $ostype
    set cmd [concat $cmd $filelist]

    catch {eval exec -keepnewline $cmd} output

    return $output
}

########################################################################
# floppyConvCmdCompose() -	Compose the command to convert a floppy
#                               image to something else (used in both
#                               saving and downloading.
########################################################################
proc floppyConvCmdCompose {num format filename} {

    global floppyType
    global floppyImage
    global floppyUploadFile
    global floppyFiles
    global floppyFilesOS
    global OSListFile
    global OSListFS

    if { $floppyType($num) == "image" } {
	set cmd [floppySaveImageCmd $floppyImage($num) $format $filename]

    } elseif { $floppyType($num) == "upload" } {
	set cmd [floppySaveImageCmd $floppyUploadFile($num) $format $filename]

    } else {
	set cmd [floppySaveFilesCmd $OSListFile($floppyFilesOS($num)) \
	                              $OSListFS($floppyFilesOS($num)) \
	                              $floppyFiles($num) $format $filename]
    }

    return $cmd
}

########################################################################
# floppyOpen() -	Opens the given floppy, preparing to download
#                       it to the SVD. 
#         NOTE - can only work on one floppy at a time...I got lazy.
#         NOTE - assumes that the file exists
#         NOTE - the whole silly floppy is read in at once!
########################################################################
proc floppyOpen {num} {
    global Floppy

    set cmd [floppyConvCmdCompose $num "" ""]

    # NOTE - should have a catch here

    set Floppy(fd) [open $cmd "r"]
    fconfigure $Floppy(fd) -encoding binary -translation binary

    set Floppy(svdversion) [gets $Floppy(fd)]
    set Floppy(sectors) [gets $Floppy(fd)]
    set Floppy(tracks) [gets $Floppy(fd)]
    set Floppy(sides) [gets $Floppy(fd)]
    set Floppy(secsize) [gets $Floppy(fd)]
    set Floppy(wprot) [gets $Floppy(fd)]
    set Floppy(data) [read $Floppy(fd)]
    
    close $Floppy(fd)
}

########################################################################
# floppySaveImageCmd() - Composes a command to save the given image to
#                        the given file.  If the file is blank, or not
#                        given, then the command comes back without a
#                        redirection of the output data.
########################################################################
proc floppySaveImageCmd {imageFile outputFormat outputFile} {
    global floppy_tosvd
    global floppy_fstool

    set ignoreCRC  "-C"

    if {$outputFile != ""} {
	set cmd "$floppy_tosvd"
    } else {
	set cmd "|$floppy_tosvd"
    }

    lappend cmd $ignoreCRC
	
    if {$outputFormat != ""} {
	lappend cmd "-o"
	lappend cmd $outputFormat
    }

    lappend cmd $imageFile

    if {$outputFile != ""} {
	lappend cmd ">"
	lappend cmd "$outputFile"
    }

    return $cmd
}

########################################################################
# floppySaveFilesCmd() - Does the saving of a set of files and OS to the
#                        given file.
########################################################################
proc floppySaveFilesCmd {osFile osType fileList outputFormat outputFile} {
    global floppy_tosvd
    global floppy_fstool

    if {$fileList == ""} {
	return [floppySaveImageCmd $osFile $outputFormat $outputFile]
    } else {

	if {$outputFile != ""} {
	    set cmd "$floppy_fstool"
	} else {
	    set cmd "|$floppy_fstool"
	}

	lappend cmd "-A"
	lappend cmd "-f"
	lappend cmd $osFile
	lappend cmd "-F"
	lappend cmd $osType

	if {$outputFormat != ""} {
	    lappend cmd "-o"
	    lappend cmd $outputFormat
	}

	set cmd [concat $cmd $fileList]

	if {$outputFile != ""} {
	    lappend cmd ">"
	    lappend cmd $outputFile
	}

	return $cmd
    }
}


proc floppyClose {} {
    global Floppy

    close $Floppy(fd)
}

########################################################################
# proc floppyNumber() -	Returns a byte indicating the floppy disk number
#                       possibly encoded with the write-protect field.
#                       The 3rd bit is set if write-protected.
#              NOTE - the floppy type is checked for "h17" which indicates
#                   that a floppy swap may occur if the option is set.
########################################################################
proc floppyNumber {num} {
    global FloppyType
    global floppyProtected

    set xmitnum $num

    # swap the floppy numbers if the option is applicable

    if { $FloppyType == "h17" } {
	if { [iniGetAttribute "Program" "H8Swap"] } {
	    if { $xmitnum == 2 } {
		set xmitnum 0
	    } elseif { $xmitnum == 0 } {
		set xmitnum 2
	    }
	}
    }

    if { $floppyProtected($num) != 0 } {
	set xmitnum [expr $xmitnum + 4]
    }

    return [binary format c $xmitnum]
}

########################################################################
# floppySectors() -	Returns a byte indicating the number of sectors.
########################################################################
proc floppySectors {} {
    global Floppy
    return [binary format c $Floppy(sectors)]
}

########################################################################
# floppySectorsString() -	Returns a string indicating sectors.
########################################################################
proc floppySectorsString {} {
    global Floppy
    return $Floppy(sectors)
}

########################################################################
# floppyTracks() -	Returns a byte indicating the number of tracks.
########################################################################
proc floppyTracks {} {
    global Floppy
    return [binary format c $Floppy(tracks)]
}

########################################################################
# floppySides() -	Returns a byte indicating the number of sides
########################################################################
proc floppySides {} {
    global Floppy
    return [binary format c $Floppy(sides)]
}

########################################################################
# floppySecSize() -	Returns a byte indicating the sector size.
########################################################################
proc floppySecSize {} {
    global Floppy
    return [binary format c $Floppy(secsize)]
}


########################################################################
# floppyData() -	Returns a subset of the binary data from the
#                       floppy as needed.  Returns 0 if the range is
#                       bad.
########################################################################
proc floppyData {start amount} {
    global Floppy
    return [string range $Floppy(data) $start [expr $start + $amount - 1]]
}

proc floppyDataSize {} {
    global Floppy
    return [string length $Floppy(data)]
}
